using System;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.Web.UI.HtmlControls;
using System.Collections;
using System.ComponentModel;
using System.Text;

using Cuyahoga.Core.Domain;
using Cuyahoga.Web.UI;
using Cuyahoga.Web.Util;

namespace Cuyahoga.ServerControls.Navigation
{
	/// <summary>
	/// The Menu class renders the complete sitemap as nested unordered lists. When styled with a proper css,
	/// it can be transformed into a nice flyout menu.
	/// </summary>
	[DefaultProperty("Text"), 
		ToolboxData("<{0}:Menu runat=server></{0}:Menu>")]
	public class Menu : System.Web.UI.WebControls.WebControl
	{
		private string _menuCss;
		private PageEngine _page;

		public string MenuCss
		{
			get { return this._menuCss; }
			set { this._menuCss = value; }
		}

		/// <summary>
		/// Create the menu controls.
		/// </summary>
		protected override void CreateChildControls()
		{
			if (this.Page is PageEngine)
			{
				this._page = (PageEngine)this.Page;
				BuildNavigationTree();
				// Set the default menu Css when none is set (in the Template css dir).
				if (this._menuCss == null)
				{
					this._menuCss = this._page.ActiveNode.Template.BasePath + "/Css/Menu.css";
				}
				this._page.RegisterStylesheet("menucss", this.Page.ResolveUrl(this._menuCss));
//				// Register hover behaviour for IE
//				StringBuilder sb = new StringBuilder(); 
//				sb.Append("<!--[if IE]>\n");
//				sb.Append("<style type=\"text/css\" media=\"screen\">\n");
//				sb.Append("/* csshover.htc available for download from: http://www.xs4all.nl/~peterned/csshover.html */\n");
//				sb.Append("body{behavior:url(" + this._page.ResolveUrl("~/csshover.htc") + ");}\n");
//				sb.Append("</style>\n");
//				sb.Append("<![endif]-->\n");
//				string hoverStyle = sb.ToString();
//				this._page.RegisterClientScriptBlock("hoverStyle", hoverStyle);
				base.CreateChildControls ();
			}
			else
			{
				throw new Exception("The menu control is not embedded in a page of the type Cuyahoga.Web.UI.PageEngine.");
			}
		}

		/// <summary> 
		/// Render this control to the output parameter specified.
		/// </summary>
		/// <param name="output"> The HTML writer to write out to </param>
		protected override void Render(HtmlTextWriter output)
		{
			base.Render(output);
		}

		private void BuildNavigationTree()
		{
			HtmlGenericControl menuContainer = new HtmlGenericControl("div");
			menuContainer.Attributes.Add("id", "menu");
			HtmlGenericControl mainList = new HtmlGenericControl("ul");
			mainList.Controls.Add(BuildListItemFromNode(this._page.RootNode));
			if (this._page.RootNode.ChildNodes.Count > 0)
			{
				mainList.Controls.Add(BuildListFromNodes(this._page.RootNode.ChildNodes));
			}
			if (this._page.CuyahogaUser != null 
				&& this._page.CuyahogaUser.HasPermission(AccessLevel.Administrator))
			{
				HtmlGenericControl listItem = new HtmlGenericControl("li");
				HyperLink hpl = new HyperLink();
				hpl.NavigateUrl = this._page.ResolveUrl("~/Admin");
				hpl.Text = "Admin";
				listItem.Controls.Add(hpl);
				mainList.Controls.Add(listItem);
			}
			menuContainer.Controls.Add(mainList);
			this.Controls.Add(menuContainer);
		}

		private HtmlControl BuildListFromNodes(IList nodes)
		{
			HtmlGenericControl list = new HtmlGenericControl("ul");
			foreach (Node node in nodes)
			{
				if (node.ShowInNavigation && node.ViewAllowed(this._page.CuyahogaUser))
				{
					HtmlControl listItem = BuildListItemFromNode(node);
					if (node.ChildNodes.Count > 0)
					{
						listItem.Controls.Add(BuildListFromNodes(node.ChildNodes));
					}
					list.Controls.Add(listItem);
				}
			}
			return list;
		}

		private HtmlControl BuildListItemFromNode(Node node)
		{
			HtmlGenericControl listItem = new HtmlGenericControl("li");
			HyperLink hpl = new HyperLink();
			hpl.NavigateUrl = UrlHelper.GetUrlFromNode(node);
			UrlHelper.SetHyperLinkTarget(hpl, node);
			hpl.Text = node.Title;

			if (node.Id == this._page.ActiveNode.Id)
			{
				hpl.CssClass = "selected";
			}
			listItem.Controls.Add(hpl);
			if (node.ChildNodes.Count > 0)
			{
				listItem.Attributes.Add("class", "haschildren");
			}

			return listItem;
		}
	}
}
